package TDAGrafo.VerticesDecorados;

import TDAGrafo.Exception.InvalidEdgeException;
import TDAGrafo.Exception.InvalidVertexException;

/**
 * Interface Grafo para grafos NO dirigidos.
 * @author Ctedra de Estructuras de Datos, Departamento de Cs. e Ing. de la Computacin, UNS.
 */
public interface Graph<V,E> {
	
	/**
	 * Devuelve una coleccin iterable de vrtices.
	 * @return Una coleccin iterable de vrtices.
	 */
	public Iterable<Vertex<V>> vertices();
	
	/**
	 * Devuelve una coleccin iterable de arcos.
	 * @return Una coleccin iterable de arcos.
	 */
	public Iterable<Edge<E>> edges();
	
	/**
	 * Devuelve una coleccin iterable de arcos incidentes a un vrtice v.
	 * @param v Un vrtice.
	 * @return Una coleccin iterable de arcos incidentes a un vrtice v.
	 * @throws InvalidVertexException si el vrtice es invlido.
	 */
	public Iterable<Edge<E>> incidentEdges(Vertex<V> v) throws InvalidVertexException;

	
	/**
	 * Devuelve el vrtice opuesto a un Arco E y un vrtice V.
	 * @param v Un vrtice
	 * @param e Un arco
	 * @return El vrtice opuesto a un Arco E y un vrtice V.
	 * @throws InvalidVertexException si el vrtice es invlido.
	 * @throws InvalidEdgeException si el arco es invlido.
	 */
	public Vertex<V> opposite(Vertex<V> v, Edge<E> e) throws InvalidVertexException, InvalidEdgeException;
	
	/**
	 * Devuelve un Arreglo de 2 elementos con lo vrtices extremos de un Arco e.
	 * @param  e Un arco
	 * @return Un Arreglo de 2 elementos con los extremos de un Arco e.
	 * @throws InvalidEdgeException si el arco es invlido.
	 */
	public Vertex<V>[] endvertices(Edge<E> e) throws InvalidEdgeException;
	
	/**
	 * Devuelve verdadero si el vrtice w es adyacente al vrtice v.
	 * @param v Un vrtice
	 * @param w Un vrtice
	 * @return Verdadero si el vrtice w es adyacente al vrtice v, falso en caso contrario.
	 * @throws InvalidVertexException si uno de los vrtices es invlido.
	 */
	public boolean areAdjacent(Vertex<V> v, Vertex<V> w) throws InvalidVertexException;
	
	/**
	 * Reemplaza el rtulo de v por un rtulo x.
	 * @param v Un vrtice
	 * @param x Rtulo
	 * @return El rtulo anterior del vrtice v al reemplazarlo por un rtulo x.
	 * @throws InvalidVertexException si el vrtice es invlido.
	 */
	public V replace(Vertex<V> v, V x) throws InvalidVertexException;
	
	/**
	 * Inserta un nuevo vrtice con rtulo x.
	 * @param x rtulo del nuevo vrtice
	 * @return Un nuevo vrtice insertado.
	 */
	public Vertex<V> insertVertex(V x);
	
	/**
	 * Inserta un nuevo arco con rtulo e, con vrtices extremos v y w.
	 * @param v Un vrtice
	 * @param w Un vrtice
	 * @param e rtulo del nuevo arco.
	 * @return Un nuevo arco.
	 * @throws InvalidVertexException si uno de los vrtices es invlido.
	 */
	public Edge<E> insertEdge(Vertex<V> v, Vertex<V> w, E e) throws InvalidVertexException;
	
	/**
	 * Remueve un vrtice V y retorna su rtulo.
	 * @param v Un vrtice
	 * @return rtulo de V.
	 * @throws InvalidVertexException si el vrtice es invlido.
	 */
	public V removeVertex(Vertex<V> v) throws InvalidVertexException;
	
	/**
	 * Remueve un arco e y retorna su rtulo.
	 * @param e Un arco
 	 * @return rtulo de E.
	 * @throws InvalidEdgeException si el arco es invlido.
	 */
	public E removeEdge(Edge<E> e) throws InvalidEdgeException;
}
